//
//  PreviewDataStore.swift
//  Do It
//
//  Created by Jim Dovey on 2/14/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import Foundation
import CoreData
import UIKit

class PreviewDataStore {
    static let shared = try! PreviewDataStore()
    
    let storeCoordinator: NSPersistentStoreCoordinator
    let objectModel: NSManagedObjectModel
    let viewContext: NSManagedObjectContext
    
    private init() throws {
        let folderURL = try FileManager.default
            .url(for: .cachesDirectory,
                 in: .userDomainMask,
                 appropriateFor: nil,
                 create: true)
            .appendingPathComponent("Do It")
        try ensureFolder(at: folderURL)
            
        let storeURL = folderURL
            .appendingPathComponent("Previews.db")
        do {
            try FileManager.default.removeItem(at: storeURL)
        } catch let err as NSError where err.domain == NSPOSIXErrorDomain {
            if err.code != ENOENT {
                throw err
            }
        } catch let err as NSError where err.domain == NSCocoaErrorDomain {
            if err.code != NSFileNoSuchFileError {
                throw err
            }
        }
        
        objectModel = NSManagedObjectModel.mergedModel(from: nil)!
        storeCoordinator = NSPersistentStoreCoordinator(managedObjectModel: objectModel)
        
        try storeCoordinator.addPersistentStore(
            ofType: NSSQLiteStoreType,
            configurationName: nil,
            at: storeURL,
            options: nil)
        
        viewContext = NSManagedObjectContext(concurrencyType: .mainQueueConcurrencyType)
        viewContext.persistentStoreCoordinator = storeCoordinator
        
        var importError: Error? = nil
        viewContext.performAndWait {
            do {
                try importSampleData(to: viewContext)
            }
            catch {
                importError = error
            }
        }
        
        if let error = importError {
            throw error
        }
    }
    
    func newBackgroundContext() -> NSManagedObjectContext {
        let context = NSManagedObjectContext(concurrencyType: .privateQueueConcurrencyType)
        context.persistentStoreCoordinator = storeCoordinator
        return context
    }

    var sampleItem: TodoItem {
        let request: NSFetchRequest<TodoItem> = TodoItem.fetchRequest()
        request.fetchLimit = 1
        let result = try? viewContext.fetch(request)
        return result!.first!
    }

    var sampleList: TodoItemList {
        let request: NSFetchRequest<TodoItemList> = TodoItemList.fetchRequest()
        request.fetchLimit = 1
        let result = try? viewContext.fetch(request)
        return result!.first!
    }
}

fileprivate func ensureFolder(at url: URL) throws {
    var isDir: ObjCBool = false
    if FileManager.default.fileExists(atPath: url.absoluteString,
                                      isDirectory: &isDir) {
        if isDir.boolValue { return }
        
        // This is our caches directory--we own it, we can delete it.
        try FileManager.default.removeItem(at: url)
    }
    
    try FileManager.default.createDirectory(
        at: url, withIntermediateDirectories: true, attributes: nil)
}
